breed [consumers consumer]
breed [sellers seller]

globals
[
  Nc
  Ns
  Wm
  total-sales
  CRn
]

consumers-own
[
  last-purchase     ;; last purchased product
  best-found
  Ri                ;; risky-taking which measures the willingness to purchase new products, compare with last-purchase product
  Pi_                ;; Pickiness, which measures the tendency to give negative reviews after purchasing
]

sellers-own [
  attraction
  wom       ;;word of mouth
  sales
;  sales-history
  score
]

;;;------------------;;;
;;; Setup PROCEDURES ;;;
;;;------------------;;;

to setup
  ca
  set Nc Num-of-Consumers
  set Ns Num-of-Sellers
  set Wm Weight-of-wom        ;; weight of WOM, compare with sales
  ask patches [set pcolor green - 1]
  create-consumers Nc [
    setxy random-xcor random-ycor
    set shape "person"
    set size 1.1
    setup-risk-taking
    setup-pickiness
    set color scale-color yellow Ri 0 1
    set label precision Pi_ 1
    set label-color red
  ]

  ask n-of Ns patches [   ;; to create sellers
    sprout 1 [
      set breed sellers
      set shape "house"
      set size 1.3
      set color white
      set wom 0
      set sales 0
      set score 0
      ;    set sales-history 0
    ]
  ]

  ;;for better looking
  ask sellers [
    while [any? other sellers with [distance myself < ceiling (16 / sqrt Ns)]]
    [move-to one-of patches with [not any? turtles-here]]
  ]
  ask consumers [
    move-to one-of patches with [not any? other turtles-here]
  ]

  ask consumers [set last-purchase one-of sellers]
  reset-ticks

end

to setup-risk-taking

  ;; beta distribution
  if (Ri-distribution = "Beta-model-1") [
    ;; Ri in [0, 1]
    set Ri beta-distribution alpha-1 beta-1
  ]
  ;; single market
  if (Ri-distribution = "extreme-market-1") [
    set Ri Ri-extreme
  ]

end

to setup-pickiness

  ;; beta distribution
  if (Pi-distribution = "Beta-model-2") [
    ;; Pi_ in [0, 1]
    set Pi_ beta-distribution alpha-2 beta-2
  ]
  ;; single market
  if (Pi-distribution = "extreme-market-2") [
    set Pi_ Pi-extreme
  ]

end

;;;---------------;;;
;;; GO PROCEDURES ;;;
;;;---------------;;;

to go

  if ticks > 150 [ stop ]
  ask consumers [search]
  ask sellers [
    set sales 0
  ]
  ask consumers [purchase]
  update-wom

  plot-results

  tick
end

to search ;; consumer procedure

  let sought n-of attention sellers with [self != [last-purchase] of myself]
  ask sought [set attraction Wm * wom + (1 - Wm) * norm-sales]                   ;; attraction in [0, 1]
  set best-found max-one-of sought [attraction]

end

to purchase ;; consumption procedure

  let d ([attraction] of best-found - [attraction] of last-purchase)   ;; d in [-1, 1]
  ifelse (abs d <= t)
  [
    ifelse (random-float 1 <= Ri) [
      consume-with best-found
      face best-found fd 1.5
    ]
    [consume-with last-purchase
      ;      if ticks = 0 [move-to one-of [neighbors] of history]
     if not member? my-point [neighbors] of last-purchase [move-to one-of [neighbors] of last-purchase]]
     ]
  [
    ifelse (d > 0)
    [
      consume-with best-found
      face best-found fd 1.5
    ]
    [
      consume-with last-purchase
      ;      if ticks = 0 [move-to one-of [neighbors] of history]
      if not member? my-point [neighbors] of last-purchase [move-to one-of [neighbors] of last-purchase]]
    ]

  ask sellers [
    set label sum [count consumers-here] of neighbors
    set label-color blue
  ]

end

to consume-with [partner]  ;; evaluation procedure + update sales

  ask partner [
    set sales sales + 1
;    set sales-history sales-history + 1
  ]
  let Ps Pi_
  ask partner [
    ifelse (random-float 1 <= Ps) [set score score - 1] [set score score + 1]
  ]
  set last-purchase partner ; change history to last-purchase

end

To update-wom

  ask sellers [
    set wom norm-score
   ; print wom
  ]
end

;;;-----------------;;;
;;; PLOT PROCEDURES ;;;
;;;-----------------;;;

to plot-results

  let sorted-sales sort [sales] of sellers
  set sorted-sales reverse sorted-sales
  set total-sales sum sorted-sales

  let first-n-sales 0
  let index 0
  repeat 1 [
    set first-n-sales first-n-sales + item index sorted-sales
    set index index + 1
  ]
  set CRn first-n-sales / total-sales * 100
  ;  show CRn  ; comparison level is : n / n-sellers
;  output-print word count sellers with [sales = 0] "sellers have no sales"
  output-print word "most popular seller is " max-one-of sellers [attraction]
;  output-print word "attraction is " [attraction] of max-one-of sellers [attraction]
;  output-print word "sales is " [sales] of max-one-of sellers [attraction]
 ; output-print word "WOM is " [wom] of max-one-of sellers [attraction]

end

;;;-------------------;;;
;;; Report PROCEDURES ;;;
;;;-------------------;;;

to-report beta-distribution [alpha beta]       ;; beta distribution

  let XX random-gamma alpha 1
  let YY random-gamma beta 1
  ifelse (XX + YY = 0) [report 0][report XX / (XX + YY)]

end

to-report norm-score ;; normalize score to the range of [0, 1] = wom

  let score-max max [score] of sellers
  let score-min min [score] of sellers
  ifelse score-max = score-min
  [report 0.5]
  [report 1 / (score-max - score-min) * (score - score-min)]

end

to-report norm-sales ;; normalize sales to the range of [0, 1]

  let sales-max max [ sales] of sellers
  let sales-min min [ sales] of sellers
  ifelse sales-max = sales-min
  [report 0.5]
  [report 1 / (sales-max - sales-min) * (sales - sales-min)]

end

to-report my-point

  report patch-at 0 0

end

;;;-----------------------------;;;
;;;   Programing by Teng Li     ;;;
;;;-----------------------------;;;
@#$#@#$#@
GRAPHICS-WINDOW
344
22
765
444
-1
-1
12.52
1
10
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
41
20
110
53
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
204
20
274
53
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
167
104
273
137
attention
attention
1
50
3.0
1
1
NIL
HORIZONTAL

SLIDER
38
62
273
95
Num-of-Consumers
Num-of-Consumers
0
500
300.0
10
1
NIL
HORIZONTAL

SLIDER
38
104
158
137
Num-of-Sellers
Num-of-Sellers
0
100
30.0
1
1
NIL
HORIZONTAL

SLIDER
37
149
159
182
Weight-of-wom
Weight-of-wom
0
1
1.0
0.1
1
NIL
HORIZONTAL

PLOT
776
25
1047
304
CR1
NIL
NIL
0.0
10.0
0.0
100.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot CRn"

MONITOR
1059
28
1119
73
CR1 now
crn
5
1
11

BUTTON
123
20
189
53
NIL
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

OUTPUT
775
353
1080
423
12

MONITOR
783
428
866
473
max sales
total-sales * CRn / 100
0
1
11

SLIDER
8
266
152
299
alpha-1
alpha-1
0
10
10.0
0.1
1
NIL
HORIZONTAL

SLIDER
10
307
153
340
beta-1
beta-1
0
10
10.0
0.1
1
NIL
HORIZONTAL

SLIDER
178
267
322
300
alpha-2
alpha-2
0
10
10.0
0.1
1
NIL
HORIZONTAL

SLIDER
178
307
322
340
beta-2
beta-2
0
10
10.0
0.1
1
NIL
HORIZONTAL

CHOOSER
9
209
148
254
Ri-distribution
Ri-distribution
"Beta-model-1" "extreme-market-1"
0

CHOOSER
180
211
318
256
Pi-distribution
Pi-distribution
"Beta-model-2" "extreme-market-2"
0

SLIDER
171
148
277
181
t
t
0
1
0.4
0.05
1
NIL
HORIZONTAL

TEXTBOX
116
347
239
395
all alpha and beta should be larger than 0
12
0.0
1

TEXTBOX
357
450
688
502
1. Consumer color from black to white shows the degree of curiosity from low to high;\n2. Label shows the pickness level from 0 to 1
12
0.0
1

INPUTBOX
23
348
105
408
Ri-extreme
0.0
1
0
Number

INPUTBOX
231
347
311
407
Pi-extreme
0.0
1
0
Number

TEXTBOX
779
322
1021
348
print max-one-of sellers [attraction]:
12
0.0
1

MONITOR
879
429
955
474
max WOM
precision [wom] of max-one-of sellers [attraction] 2
17
1
11

MONITOR
967
429
1064
474
max attraction
[attraction] of max-one-of sellers [attraction]
17
1
11

PLOT
23
421
311
571
Type distribution
Ri/Pk
NIL
0.0
1.05
0.0
0.0
true
true
"" ""
PENS
"Risky" 0.05 0 -1184463 true "" "histogram [Ri] of Consumers"
"Picky" 0.05 0 -13345367 true "" "histogram [Pi_] of Consumers"

@#$#@#$#@
## WHAT IS IT?

This model explores the dynamic relationship between consumer types and market concentration, with a specific focus on the interplay between word-of-mouth effects and social influence. It integrates multiple elements, including consumers, sellers, bounded rationality, social networks, and interpersonal interactions. The model investigates how consumers adapt their purchasing decisions based on evolving influences, such as product attractiveness, sales, and social feedback from peers. These adaptations are governed by a set of behavioral rules, providing insights into the mechanisms shaping market dynamics.


## HOW IT WORKS


### DURING GO

The consumer purchase process is summarized into three stages: search, consumption, and evaluation.

Search Stage: Consumers begin by seeking information about available options to meet their needs. The extent of consumer attention determines the breadth of their search; consumers with higher attention are more likely to explore a wider range of alternatives before making a purchase, while those with lower attention focus on fewer options.

Consumption Stage: Consumers make purchasing decisions based on perceived value and how effectively a product meets their needs. This stage is influenced by consumer types: risk-taking consumers tend to prioritize new or unfamiliar products within the set of alternatives, whereas conservative consumers prefer familiar, established options.

Evaluation Stage: After purchasing, consumers evaluate their satisfaction with the product or service based on their experience. This evaluation often manifests in feedback, such as online reviews, contributing to the overall word-of-mouth dynamic.

A consumer (“person”) will be physically close to that seller (“house”) every time he or she buys a product from that seller.

### SIMULATION OUTPUT

The **"CR1"** Monitor displays the current value of CR1, indicating the cumulative market share of the top sellers in real-time.

**"CR1-now"** is a metric that aggregates the percentage market shares held by the leading seller within a specific market at each step.


## HOW TO USE IT


* "Num-of-Consumers" determines the quantity of consumers in the simulation, with a default setting of 300.

* "Num-of-Sellers" sets the number of sellers in the simulation, defaulting to 30.

* "attention" denotes the pre-purchase search capacity, specifying the number of sellers consumers explore within the market during a defined timeframe, defaulting to 3.

* "Weight-of-wom" specifies the relative importance of word-of-mouth compared to sales in determining product attraction.

* "t" represents the tolerance threshold for differences in product satisfaction, ranging from 0 to 1. When t=0, all consumers purchase only the product with the highest attraction. Conversely, when t=1, consumers make their purchase decisions solely based on their personal risk preferences.

* "Ri-distribution" (risk preference distribution) and "Pi-distribution" (picky trait distribution) define the variability of consumer traits in the market. Two distribution forms are available:

1. Beta distribution, selected via "Beta-model-1" (for risk preferences) or "Beta-model-2" (for picky traits). The shape of the beta distribution is controlled by the parameters "alpha-1" and "beta-1" (or "alpha-2" and "beta-2" for picky traits).

2. Extreme distribution, chosen via "extreme-market-1" or "extreme-market-2," assumes a market with only one consumer type. This is achieved by setting "Ri-extreme" (or "Pi-extreme") to 0 or 1.


## THINGS TO NOTICE


Firstly, it's important to note that the value of "attention" must not exceed the value of "num-sellers" and all alpha and beta should be larger than 0.

Secondly, it is crucial to emphasize that, during the model's operation, all other parameters must remain constant and unchanged.


## THINGS TO TRY


Seveal things can be tyied, for example:

* Adjusting Parameters:

Experiment with the "Weight-of-wom" slider to observe how the relative importance of word-of-mouth versus sales affects product attraction and market dynamics.

Vary the "t" value to see how different tolerance thresholds influence consumer decision-making, from selecting the most attractive product to prioritizing personal risk preferences.

* Exploring Distributions:

Select "Beta-model-1" or "Beta-model-2" and modify the "alpha" and "beta" values to investigate how different shapes of the beta distribution affect consumer behavior.

Switch to "extreme-market-1" or "extreme-market-2" and set "Ri-extreme" or "Pi-extreme" to 0 or 1 to simulate markets with only one consumer type.


## EXTENDING THE MODEL


The model can be extended by exploring additional features, implement more complex behaviors, or apply the model to new scenarios. For example:

* Consumer Heterogeneity: 

Introduce additional consumer traits, such as price sensitivity or brand loyalty, to capture a more nuanced range of decision-making behaviors.

Implement a dynamic consumer type system, where individuals can shift between risk-taking and conservative behaviors based on their experiences or peer influence.

* Product Differentiation:

Expand the model to include multiple attributes for products, such as price, quality, and sustainability. This would allow for more complex trade-offs in consumer decision-making.

Simulate product innovation by allowing sellers to introduce new products with unique features over time.


## NETLOGO FEATURES


In this model, consmuers are denoted by "person", and sellers are denoted by "house".
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.4.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="E1-CT" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Weight-of-wom">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;extreme-market-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-distribution">
      <value value="&quot;extreme-market-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="t">
      <value value="0.6"/>
      <value value="0.7"/>
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="E1-RT" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Weight-of-wom">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;extreme-market-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-distribution">
      <value value="&quot;extreme-market-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="t">
      <value value="0.6"/>
      <value value="0.7"/>
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="E1-CP" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Weight-of-wom">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;extreme-market-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-distribution">
      <value value="&quot;extreme-market-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-extreme">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="t">
      <value value="0.6"/>
      <value value="0.7"/>
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="E1-RP" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Weight-of-wom">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;extreme-market-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-distribution">
      <value value="&quot;extreme-market-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-extreme">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="t">
      <value value="0.6"/>
      <value value="0.7"/>
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="E1-CT-wom" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Weight-of-wom">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;extreme-market-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-distribution">
      <value value="&quot;extreme-market-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="t">
      <value value="0.7"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="E1-RT-wom" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Weight-of-wom">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;extreme-market-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-distribution">
      <value value="&quot;extreme-market-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="t">
      <value value="0.7"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="E1-CP-wom" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Weight-of-wom">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;extreme-market-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-distribution">
      <value value="&quot;extreme-market-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-extreme">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="t">
      <value value="0.7"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="E1-RP-wom" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Weight-of-wom">
      <value value="0"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;extreme-market-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-distribution">
      <value value="&quot;extreme-market-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-extreme">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="t">
      <value value="0.7"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="E2-distribution-Tolerant" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="1"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Weight-of-wom">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;Beta-model-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="1"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-distribution">
      <value value="&quot;extreme-market-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="t">
      <value value="0.7"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="E2-distribution-Picy" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="1"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Weight-of-wom">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;Beta-model-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="1"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-distribution">
      <value value="&quot;extreme-market-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-extreme">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="t">
      <value value="0.7"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="E2-distribution-Conservative" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="1"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Weight-of-wom">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;extreme-market-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="1"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-distribution">
      <value value="&quot;Beta-model-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="t">
      <value value="0.7"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="E2-distribution-Risky" repetitions="20" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="1"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Weight-of-wom">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;extreme-market-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="1"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-distribution">
      <value value="&quot;Beta-model-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="t">
      <value value="0.7"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="E3-complex-wom" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Weight-of-wom">
      <value value="0"/>
      <value value="0.5"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;Beta-model-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-distribution">
      <value value="&quot;Beta-model-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pk-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="t">
      <value value="0.7"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="E4-complex" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Weight-of-wom" first="0" step="0.1" last="1"/>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;Beta-model-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pi-distribution">
      <value value="&quot;Beta-model-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pi-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <steppedValueSet variable="t" first="0" step="0.1" last="1"/>
  </experiment>
  <experiment name="E4-CT" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Weight-of-wom" first="0" step="0.1" last="1"/>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;extreme-market-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pi-distribution">
      <value value="&quot;extreme-market-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pi-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <steppedValueSet variable="t" first="0" step="0.1" last="1"/>
  </experiment>
  <experiment name="E4-RT" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Weight-of-wom" first="0" step="0.1" last="1"/>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;extreme-market-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pi-distribution">
      <value value="&quot;extreme-market-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pi-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <steppedValueSet variable="t" first="0" step="0.1" last="1"/>
  </experiment>
  <experiment name="E4-RP" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Weight-of-wom" first="0" step="0.1" last="1"/>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;extreme-market-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pi-distribution">
      <value value="&quot;extreme-market-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pi-extreme">
      <value value="1"/>
    </enumeratedValueSet>
    <steppedValueSet variable="t" first="0" step="0.1" last="1"/>
  </experiment>
  <experiment name="E4-CP" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="150"/>
    <metric>CRn</metric>
    <enumeratedValueSet variable="alpha-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="alpha-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-extreme">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="attention">
      <value value="3"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Weight-of-wom" first="0" step="0.1" last="1"/>
    <enumeratedValueSet variable="Num-of-Sellers">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Ri-distribution">
      <value value="&quot;extreme-market-1&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-1">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="beta-2">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Num-of-Consumers">
      <value value="300"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pi-distribution">
      <value value="&quot;extreme-market-2&quot;"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Pi-extreme">
      <value value="1"/>
    </enumeratedValueSet>
    <steppedValueSet variable="t" first="0" step="0.1" last="1"/>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
